/*

 Copyright (c) 1997, Kevin Lang, NEC Research Institute, 
 See NECI license below.

  This is a blue-fringe evidence-driven state merging algorithm, as
  described in the 1998 ICGI paper of Lang, Pearlmutter, and Price.
  The details of the outer loop come from Juille and Pollack's paper.

  This program can solve Abbadingo problems 1, 2, 3, R, 4, 6, and S.

  How to compile:

  gcc -Wall -O4 -o red-blue red-blue.c data-structures-1.c -DUSE_FREE

  or 

  gcc -Wall -g -o red-blue red-blue.c data-structures-1.c -DUSE_GC ~/lib/gc.a 
  
  You can also specify -DNOISY to make the program print out more stuff.

  The program seems to be a bit faster in the garbage collected version,
  using Boehm's very fine garbage collector for C.
  
  */


/************************************************************************/



/*
  
 Copyright (c) 1997,1998 NEC Research Institute Inc.  All rights reserved. 
 
 Permission to use, copy, modify, and distribute this software and its
 associated documentation for non-commercial purposes is hereby
 granted, provided that the above copyright notice appears in all
 copies, derivative works or modified versions of the software and any
 portions thereof, and that both the copyright notice and this
 permission notice appear in the documentation.  NEC Research Institute
 Inc. shall be given a copy of any such derivative work or modified
 version of the software and NEC Research Institute Inc. and its
 affiliated companies (collectively referred to as NECI) shall be
 granted permission to use, copy, modify and distribute the software
 for internal use and research.  The name of NEC Research Institute
 Inc. and its affiliated companies shall not be used in advertising or
 publicity related to the distribution of the software, without the
 prior written consent of NECI.  All copies, derivative works or
 modified versions of the software shall be exported or reexported in
 accordance with applicable laws and regulations relating to export
 control.  This software is experimental.  NECI does not make any
 representations regarding the suitability of this software for any
 purpose and NECI will not support the software.  
 
 THE SOFTWARE IS PROVIDED AS IS.  NECI DOES NOT MAKE ANY WARRANTIES
 EITHER EXPRESS OR IMPLIED WITH REGARD TO THE SOFTWARE.  NECI ALSO
 DISCLAIMS ANY WARRANTY THAT THE SOFTWARE IS FREE OF INFRINGEMENT OF
 ANY INTELLECTUAL PROPERTY RIGHTS OF OTHERS.  NO OTHER LICENSE EXPRESS
 OR IMPLIED IS HEREBY GRANTED. NECI SHALL NOT BE LIABLE FOR ANY
 DAMAGES, INCLUDING GENERAL, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 DAMAGES, ARISING OUT OF THE USE OR INABILITY TO USE THE SOFTWARE.

*/




/************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <sys/types.h>
#include <sys/time.h>
#include <setjmp.h>

#include "data-structures-1.h"


#ifdef USE_GC
#include "../../include/gc.h"
#else
#include <malloc.h>
#endif


#ifndef dotimes
#define dotimes(var,lim) for (var=0;var<(lim);var++) 
#define do_cdrs(var,srclist) for (var=srclist;var!=NULL;var=CDR(var))
#define MAX(a,b) (((a)>(b)) ? (a) : (b))
#define MIN(a,b) (((a)<(b)) ? (a) : (b))
#define ABS(zz) ((zz)<0 ? -1*(zz) : (zz))
#endif

#ifndef boolean
#define boolean int
#define false 0
#define true 1
#endif




/***************************************************************/


#define S_white 0
#define S_blue  1
#define S_red   2

int alphabet_size = 2;

/* Not every slot in a node needs to be valid throughout the
   computation.  However, father needs to be correct while a node is
   blue or white, depth needs to be correct while a node is blue or
   red, and enemies needs to be correct while a node is blue.  */

typedef struct tree_node
{
  int gloid_slot;  /* an identifying number */
  int label_slot;  
  int depth_slot;  
  int color_slot;  
  I_Table * enemies_slot;               /* table of incompatible red nodes */
  struct tree_node * father_slot;       /* backpointer to father */
  struct tree_node ** child_array_slot; /* pointers to children */
} treeNode;


#define NONDEF NULL

#define nondef_p(tn)         ((tn)==NONDEF)

#define gloid(tn)            ((tn)->gloid_slot)
#define get_label(tn)        ((tn)->label_slot)
#define get_depth(tn)        ((tn)->depth_slot)
#define get_color(tn)        ((tn)->color_slot)
#define get_enemies(tn)      ((tn)->enemies_slot)
#define get_father(tn)       ((tn)->father_slot)
#define get_child(tn,i)      ((tn)->child_array_slot[(i)])

#define set_label(tn,new)    ((tn)->label_slot=(new))
#define set_depth(tn,new)    ((tn)->depth_slot=(new))
#define set_color(tn,new)    ((tn)->color_slot=(new))
#define set_enemies(tn,new)  ((tn)->enemies_slot=(new))
#define set_father(tn,new)   ((tn)->father_slot=(new))
#define set_child(tn,i,new)  ((tn)->child_array_slot[(i)]=(new))

inline boolean label_defined (int lab)
{
  return (! (-1 == lab));
}


/***************************************************************/


int global_node_counter = 0;


treeNode * make_node (int depth, treeNode * father)
{
  int i;
  treeNode * tn = (treeNode *) my_malloc (sizeof(treeNode));
  treeNode ** ch = (treeNode **) my_malloc (alphabet_size*sizeof(treeNode *));

  tn->gloid_slot = global_node_counter++;
  tn->label_slot = -1;
  tn->depth_slot = depth;
  tn->color_slot = S_white;
  tn->father_slot = father;
  tn->enemies_slot = NULL; /* create table when node turns blue */
  tn->child_array_slot = ch;
  dotimes (i, alphabet_size) ch[i] = NONDEF;

  return tn;
}

/*********************************************************************/

void depth_first_print (treeNode * curnode)
{
  int i;
  if (nondef_p (curnode)) printf("?");
  else {
    printf("[%d %d ", gloid (curnode), get_label (curnode));
    dotimes (i, alphabet_size)
      depth_first_print(get_child(curnode, i));
    printf("]");
  }
}


void augment_tree (treeNode * curnode, IP_Pair * str, int label)
{
  if (str == NULL)
    set_label(curnode,label);
  else {
    int cursym = ICAR(str);
    if (nondef_p(get_child(curnode,cursym)))
      set_child(curnode,cursym,make_node(1+get_depth(curnode),curnode));
    augment_tree(get_child(curnode,cursym), CDR(str), label);
  }
}




/*********************************************************************/


treeNode * new_build_tree (PP_Pair * example_list)
{
  treeNode * root = make_node (0, NONDEF);
  PP_Pair * l = example_list;
  while (l != NULL) {
    IP_Pair * example = PCAR(l);
    augment_tree(root,CDR(example),ICAR(example));
    l = CDR(l); 
  }
  return root;
}

int read_cform_called_yet = 0;
int length_of_longest_string = -1;


PP_Pair * read_cform_trainset (char * name)
{
  FILE * infile = fopen (name, "r");
  int n_strings, i, j;
  PP_Pair * accumulate = NULL;
  PP_Pair * result;

  assert(read_cform_called_yet == 0);
  read_cform_called_yet = 1;

  if (infile == NULL) {
    printf("file %s not found\n", name);
    exit(-1);
  }

  fscanf(infile, "%d", &n_strings);
  fscanf(infile, "%d", &alphabet_size);

  dotimes (i, n_strings) {
    int lab, strlen, itmp;
    IP_Pair * rb2 = NULL;
    fscanf(infile, "%d", &lab);
    fscanf(infile, "%d", &strlen);
    dotimes (j, strlen) {
      fscanf(infile, "%d", &itmp);
      rb2 = ilist_cons(itmp, rb2);
    }
    length_of_longest_string = MAX(strlen,length_of_longest_string);
    accumulate = plist_cons (ilist_cons(lab,ilist_reverse (rb2)), accumulate);
#ifdef USE_FREE
    ilist_dispose(rb2);
#endif
  }
  fclose (infile);
  result = plist_reverse (accumulate);
#ifdef USE_FREE
    plist_dispose(accumulate);
#endif
  return result;
}



/*********************************************************************/


/* note: for unknown labels and transitions, we are currently printing
   zeros instead of -1 and ? like we used to do.  */


void fancy_print_tree (treeNode * start_node)
{
  I_Table * seen = make_itable ();
  I_Table * id_table = make_itable ();
  P_Queue * todo = make_pqueue ();
  int count = 0;
  int i;

  pqueue_push (todo, (void *) start_node);

  while (!pqueue_empty_p(todo)) {
    treeNode * curnode = pqueue_pop(todo);
    if (!itable_test(id_table,gloid(curnode))) {
      itable_set (id_table, gloid(curnode), count++);
      dotimes (i, alphabet_size) {
	treeNode * x = get_child(curnode,i);
	if (!nondef_p(x)) pqueue_push(todo,x);
      }
    }
  }

  printf("%d %d\n", count, alphabet_size);

  pqueue_push (todo, (void *) start_node);

  while (!pqueue_empty_p(todo)) {
    treeNode * curnode = pqueue_pop(todo);
    int lab = get_label(curnode);
    if (!itable_test_and_set(seen,gloid(curnode))) {
      printf("%d %d ",
	     itable_lookup(id_table, gloid(curnode)),
	     (lab == -1) ? 0 : lab);
      dotimes (i, alphabet_size) {
	treeNode * x = get_child(curnode,i);
	if (nondef_p(x))
	  printf("0 ");
	else {
	  pqueue_push(todo,x);
	  printf("%d ", itable_lookup(id_table,gloid(x)));
	}
      }
      printf("\n");
    }
  }

#ifdef USE_FREE
  deallocate_pqueue (todo);
  deallocate_itable (seen);
  deallocate_itable (id_table);
#endif


}



/*********************************************************************/



/* not currently used */

PP_Pair * make_breadth_first_node_list (treeNode * start_node)
{
  I_Table * seen = make_itable ();
  P_Queue * todo = make_pqueue ();
  PP_Pair * outlist = NULL;
  PP_Pair * result;
  int i;

  pqueue_push (todo, (void *) start_node);

  while (!pqueue_empty_p(todo)) {
    treeNode * curnode = pqueue_pop(todo);
    if (!itable_test(seen,gloid(curnode))) {
      itable_set (seen, gloid(curnode), 1);
      outlist = plist_cons (curnode, outlist);
      dotimes (i, alphabet_size) {
	treeNode * x = get_child(curnode,i);
	if (!nondef_p(x)) pqueue_push(todo,x);
      }
    }
  }

  result = plist_reverse (outlist);

#ifdef USE_FREE
  deallocate_pqueue (todo);
  deallocate_itable (seen);
  plist_dispose (outlist);
#endif

  return result;
}


/************************************************************/


/* This routine corrects the depth and father fields, which may have
   been invalidated by earlier merges.  

   Important: to save time, we are only going deep enough to fix
   red and blue nodes.  */


void walk_graph_fixing_depths_and_backpointers (treeNode * start_node)
{
  I_Table * seen = make_itable ();
  P_Queue * todo = make_pqueue ();
  int i;

  pqueue_push (todo, (void *) make_ipp_triple (0, start_node, NULL));

  while (!pqueue_empty_p(todo)) {
    IPP_Triple * pear = pqueue_pop(todo);
    int curdepth = pear->i1;
    treeNode * curnode = pear->p2;
    treeNode * curdad  = pear->p3;
#ifdef USE_FREE    
    my_free (pear, sizeof(IPP_Triple));
#endif
    if (!itable_test(seen,gloid(curnode))) {
      itable_set (seen, gloid(curnode), 1);
      set_depth  (curnode, curdepth);
      set_father (curnode, curdad);
      if (S_red == get_color(curnode)) { /* visit children of red nodes */
	dotimes (i, alphabet_size) {
	  treeNode * x = get_child(curnode,i);
	  if (!nondef_p(x))
	    pqueue_push(todo,
			make_ipp_triple(1+curdepth, x, curnode));
	}
      }
    }
  }

#ifdef USE_FREE
  deallocate_pqueue (todo);
  deallocate_itable (seen);
#endif

}


/*********************************************************************/




int find_incoming_token (treeNode * node, treeNode * father) 
{
  int i = 0;
  int n_matches = 0;
  int matching_i = -1;

  dotimes (i, alphabet_size) {
    if (get_child(father, i)==node) {
      n_matches += 1;
      matching_i = i;
    }
  }

  if (n_matches != 1) printf("nm=%d  mi=%d\n", n_matches, matching_i);
  assert (n_matches == 1);
  return (matching_i);
}


/*********************************************************************/


#define nonnegative_p(x) ((x) >= 0)

#define ip_pair_get_i(x) (x)->i1
#define ip_pair_get_p(x) (x)->p2


#define  ipp_triple_get_i(x) (x)->i1
#define  ipp_triple_get_p(x) (x)->p2
#define  ipp_triple_get_q(x) (x)->p3

/*********************************************************************/


typedef struct walk_state
{
  int match_count;
  PP_Pair * fix_trans;
  PP_Pair * fix_label;
  jmp_buf bailout;
} walkState;


void undo_things (walkState * w);
void dont_undo_things (walkState * w);
void walkit (treeNode * r, treeNode * b, walkState * w);


/* Here we evaluate a pair of nodes for mergeability by doing a
   simultaneous walk starting at the two nodes.  The walk terminates
   because the blue node is the root of a tree.  If a labeling
   conflict is found, we return -1.  Otherwise we return the number of
   actively matching labels.  Note that during the walk we modify the
   hypothesis by copying labels and splicing branches from the blue
   side to the red side.  We keep track of these changes in the fix
   fields of the walkState data structure so that they can be undone
   if the merge fails or if the walk was being done for scoring purposes
   only.  Don't be tempted to eliminate the side effects in the latter
   case.  They are necessary to check the validity of a merge.
*/


int merging_compute_score (treeNode * r_cand, treeNode * b_cand,
			   boolean for_real_p) 
{
  walkState ws; walkState * w = &ws;
  treeNode * b_father = get_father (b_cand);
  int        b_insym  = find_incoming_token (b_cand, b_father);

  w->match_count = 0; w->fix_trans = NULL; w->fix_label = NULL;
  set_child (b_father, b_insym, r_cand);

  if (setjmp(w->bailout) == -1) {
    /* this is the return from a bailout */
    undo_things (w);
    set_child (b_father, b_insym, b_cand);
    return -1;
  }
  else {
    walkit (r_cand, b_cand, w);
    if (!for_real_p) {
      undo_things (w);
      set_child (b_father, b_insym, b_cand);
    }
    else dont_undo_things(w);
    return w->match_count;
  }
}



void walkit (treeNode * r, treeNode * b, walkState * w)
{
  int i;
  if (label_defined (get_label (b))) {
    if (label_defined (get_label (r))) {
      if (get_label (r) == get_label (b))
	w->match_count = w->match_count + 1;
      else longjmp(w->bailout, -1);
    }
    else {
      w->fix_label = plist_cons (r, w->fix_label);
      set_label (r, get_label (b));
    }
  }
  dotimes (i, alphabet_size) {
    treeNode * r_child = get_child (r, i);
    treeNode * b_child = get_child (b, i);
    if (!(nondef_p (b_child))) {
      if (!(nondef_p (r_child)))
	walkit (r_child, b_child, w);
      else { 
	w->fix_trans = plist_cons (make_ip_pair (i,r), w->fix_trans);
	set_child (r, i, b_child);
      }
    }
  }
}



void undo_things (walkState * w)
{ 
  PP_Pair * tees, * ells;
  do_cdrs (tees, w->fix_trans) {
    IP_Pair * tee = PCAR(tees);
    set_child ((treeNode *) ip_pair_get_p (tee), ip_pair_get_i (tee), NONDEF);
  }
  do_cdrs (ells, w->fix_label) {
    treeNode * ell = PCAR(ells);
    set_label (ell, -1);  
  }
#ifdef USE_FREE  
  plist_dispose(w->fix_label);
  deallocate_list_of_ip (w->fix_trans);
#endif
}


void dont_undo_things (walkState * w)
{
#ifdef USE_FREE  
  plist_dispose (w->fix_label);
  deallocate_list_of_ip (w->fix_trans);
#endif
}




/*********************************************************************/


/* 
  Here we check to see whether a pair of nodes has previously
  been found unmergeable before calling the real merge evaluator.
   */


int caching_compute_score (treeNode * r_cand,
			   treeNode * b_cand,
			   boolean for_real_p)
{
  if (itable_test (get_enemies(b_cand), gloid(r_cand))) return -1;
  else {
    int score = merging_compute_score (r_cand, b_cand, for_real_p);
    if (nonnegative_p (score)) return score;
    else {
      itable_set (get_enemies(b_cand), gloid(r_cand), true);
      return -1;
    }
  }
}






/*********************************************************************/



#ifdef NOISY

#define NOISE_1 fprintf (stderr,"%d red %d blue, %d promos %d merges, ", \
			 plist_length (pstack_contents (red_stk)), \
			 plist_length (pstack_contents (blue_stk)), \
			 plist_length (promos), \
			 plist_length (merges)); 

#define NOISE_2 fprintf (stderr, "promoting %d (d %d)\n", \
			 gloid (promoted), get_depth(promoted));

#define NOISE_3 fprintf (stderr, "merging %d %d (score %d)\n", \
			 gloid ((treeNode *) ipp_triple_get_p (cand)), \
			 gloid ((treeNode *) ipp_triple_get_q (cand)), \
			 ipp_triple_get_i (cand));

#define NOISE_4 fprintf (stderr, "done\n");

#else

#define NOISE_1
#define NOISE_2
#define NOISE_3
#define NOISE_4

#endif



/************************************************/

treeNode * nonrandom_shallowest_node (PP_Pair * inlist) 
{
  PP_Pair * work;
  treeNode * x = (treeNode *) PCAR (inlist);
  int v        = get_depth (x);
  do_cdrs (work, CDR(inlist)) {
    treeNode * y = (treeNode *) PCAR (work);
    if (get_depth (y) < v) {
      x = y;
      v = get_depth (y);
    }
  }
  return x;
}


IPP_Triple * nonrandom_highest_scoring_cand (PP_Pair * inlist) 
{
  PP_Pair * work;
  IPP_Triple * x = (IPP_Triple *) PCAR (inlist);
  int v        = ipp_triple_get_i (x);
  do_cdrs (work, CDR(inlist)) {
    IPP_Triple * y = (IPP_Triple *) PCAR (work);
    if (ipp_triple_get_i (y) > v) {
      x = y;
      v = ipp_triple_get_i (y);
    }
  }
  return x;
}


/***************************************/

#define SCALE 0.4656612875e-9  
#define myrandom(x)  ((x) * (random() * SCALE ))
#define myrandomint(low,high) ( (int) (low + (high-low+1) * myrandom(1.0)))


/* pick the shallowest node out of a list, breaking ties randomly */

treeNode * shallowest_node (PP_Pair * inlist) 
{
  PP_Pair * work;
  PP_Pair * xs = plist_cons (PCAR (inlist), NULL);
  int v         = get_depth  ((treeNode *) PCAR (inlist));
  do_cdrs (work, CDR(inlist)) {
    treeNode * y = (treeNode *) PCAR (work);
    if (get_depth (y) == v) {
      xs = plist_cons (y, xs);
    }
    else if (get_depth (y) < v) {
#ifdef USE_FREE    
    plist_dispose (xs);
#endif
      xs = plist_cons (y, NULL);
      v  = get_depth  (y);
    }
  }
  {
    int len = plist_length (xs);
    treeNode * result = plist_nth (xs, myrandomint (0,len-1));
#ifdef USE_FREE    
    plist_dispose (xs);
#endif
    return result;
  }
}


/* pick the best merge out of a list, breaking ties randomly */

IPP_Triple * highest_scoring_cand (PP_Pair * inlist) 
{
  PP_Pair * work;
  PP_Pair * xs = plist_cons (PCAR (inlist), NULL);
  int v         = ipp_triple_get_i  ((IPP_Triple *) PCAR (inlist));
  do_cdrs (work, CDR(inlist)) {
    IPP_Triple * y = (IPP_Triple *) PCAR (work);
    if (ipp_triple_get_i (y) == v) {
      xs = plist_cons (y, xs);
    }
    else if (ipp_triple_get_i (y) > v) {
#ifdef USE_FREE    
      plist_dispose (xs);
#endif
      xs = plist_cons (y, NULL);
      v  = ipp_triple_get_i  (y);
    }
  }
  {
    int len = plist_length (xs);
    IPP_Triple * result = plist_nth (xs, myrandomint (0,len-1));
#ifdef USE_FREE    
    plist_dispose (xs);
#endif
    return result;
  }
}






/************************************************/


/* 
   This routine finds the non-red children of red nodes,
   and colors them blue.
 */


P_Stack * compute_blue_nodes (P_Stack * red_stack) 
{
  int i;
  P_Stack * blue_stack = make_pstack ();
  PP_Pair * work;
  do_cdrs (work, pstack_contents (red_stack)) {
    treeNode * r = PCAR(work);
    dotimes (i, alphabet_size) {
      treeNode * c = get_child (r, i);
      if (! (nondef_p (c)) &&
	  ! (get_color (c) == S_red) &&
	  ! (plist_member_p (c, pstack_contents (blue_stack)))) {
	if (get_color(c)==S_white) /* new blue nodes need a table */
	  set_enemies (c, make_itable ());
	set_color (c, S_blue);
	pstack_push (blue_stack, c);
      }
    }
  }
  return blue_stack;
}


/************************************************/

/* 
  This routine examines all possible pairings of a red node and a blue
  node.  Pairs that could be merged get returned on a list of merge
  candidates.  Blue nodes nodes that are unmergeable with any
  red node are eligible for promotion to red status, so they
  are returned on a list of promotion candidates.
 */

void compute_merges_and_promos (PP_Pair ** return_merges,
				PP_Pair ** return_promos,
				P_Stack * red_stack,
				P_Stack * blue_stack)
{
  PP_Pair * merges = NULL;
  PP_Pair * promos = NULL;
  PP_Pair * bwork, *rwork;
  do_cdrs (bwork, pstack_contents (blue_stack)) {
    treeNode * b = PCAR(bwork);
    boolean any_merges_p = false;
    do_cdrs (rwork, pstack_contents (red_stack)) {
      treeNode * r = PCAR(rwork);
      int the_score = caching_compute_score (r, b, false);
      if (nonnegative_p (the_score)) {
	/* hack score to favor merges to shallow red nodes */   
        the_score = 100 * the_score + (99 - get_depth (r));
	merges = plist_cons (make_ipp_triple (the_score, r, b), merges);
	any_merges_p = true;
      }
    }
    if (!any_merges_p)
      promos = plist_cons (b, promos);
  }
  *return_merges = merges;
  *return_promos = promos;
}


/*********************************************************************/



#ifdef USE_FREE
#define FREE_1 deallocate_itable(get_enemies(promoted)); \
    deallocate_pstack (blue_stk); \
    plist_dispose (promos); \
    deallocate_list_of_ipp (merges); 

#define FREE_2 deallocate_itable (get_enemies((treeNode *) \
                                    ipp_triple_get_q(cand))); \
    deallocate_pstack (blue_stk); \
    plist_dispose (promos); \
    deallocate_list_of_ipp (merges);

#define FREE_3 deallocate_pstack (red_stk); \
    deallocate_pstack (blue_stk); \
    plist_dispose (promos); \
    deallocate_list_of_ipp (merges);
#else
#define FREE_1
#define FREE_2
#define FREE_3
#endif


/******************************************************************/



/* this routine contains the program's outer loop */



void red_blue_collapse (treeNode * root) 
{
  P_Stack * red_stk, * blue_stk;
  PP_Pair * promos, * merges;

  red_stk = make_pstack ();
  set_color (root, S_red); pstack_push (red_stk, root);

outer_loop:

  blue_stk = compute_blue_nodes (red_stk);

  walk_graph_fixing_depths_and_backpointers (root); 
  compute_merges_and_promos (&merges, &promos, red_stk, blue_stk);
  NOISE_1;
  if (promos != NULL) {  /* promotions have higher priority than merges */
    treeNode * promoted = shallowest_node (promos);
    NOISE_2; FREE_1;
    set_color (promoted, S_red);      /* do a promotion */
    pstack_push (red_stk, promoted);
    set_enemies (promoted, NULL);     /* red node doesn't need a table */
    goto outer_loop;
  }
  else if (merges != NULL) {
    IPP_Triple * cand = highest_scoring_cand (merges);
    merging_compute_score (ipp_triple_get_p (cand),
			   ipp_triple_get_q (cand),
			   true); 
    NOISE_3; FREE_2;
    goto outer_loop;
  }
  else {
    NOISE_4; FREE_3;
  }
}


/************************************************/


long initial_seconds;


void init_millisecond_timer ()
{
  struct timeval t_work;
  gettimeofday(&t_work, NULL);
  initial_seconds = t_work.tv_sec;
}


/* this gives a timestamp in units of milliseconds */
/* the actual time resolution may be lower */


int millisecond_timer ()
{
  struct timeval t_work;
  int n_seconds;
  gettimeofday(&t_work, NULL);
  n_seconds = t_work.tv_sec - initial_seconds;
  return(n_seconds * 1000 + t_work.tv_usec / 1000);
}





/*********************************************************************/



int main (int argc, char **argv)
{
  treeNode * pig; 
  PP_Pair * training_set;
  int t1, t2, t3, t4;

  if (argc != 2) {
    printf("usage: %s filename\n", argv[0]);
    exit(-1);
  }

  srandom(time(NULL));
  init_millisecond_timer ();

  t1 = millisecond_timer ();

  training_set = read_cform_trainset (argv[1]);

  pig = new_build_tree (training_set);

#ifdef USE_FREE
  deallocate_int_list_list (training_set);
#endif

  t2 = millisecond_timer ();

#ifdef NOISY
  fprintf(stderr, "%d nodes allocated\n", global_node_counter); 
#endif

  red_blue_collapse (pig);
  t3 = millisecond_timer ();

  fancy_print_tree (pig);  
  t4 = millisecond_timer ();

#ifdef NOISY
  my_malloc_report(); 
  fprintf(stderr, "%d msec reading, %d msec crunching, %d msec writing\n",
	  t2-t1, t3-t2, t4-t3);
#endif

  return(0);

}

