
/*
  change history:
         April 14, 1998
         fixed queue_contents
         added list_append_bang
 */

/***********************************************/

/* copyright 1996, 1997, 1998 Kevin Lang, NECI */


#include <stdio.h>
#include <assert.h>


#ifdef USE_GC
#include "../../include/gc.h"
#else
#include <malloc.h>
#endif


#include "data-structures-1.h"


#ifndef dotimes
#define dotimes(var,lim) for (var=0;var<(lim);var++) 
#endif


#ifndef do_cdrs
#define do_cdrs(var,srclist) for (var=srclist;var!=NULL;var=CDR(var))
#endif

/************************************************************/


int total_allocated = 0;
int total_freed = 0;

void my_malloc_report ()
{
  fprintf(stderr, "%d bytes allocated, %d freed, %d unfreed\n",
	  total_allocated, total_freed,
	  total_allocated- total_freed);
}


#ifdef USE_GC
inline void * my_malloc (int nbytes)
{
  total_allocated += nbytes;
  return GC_malloc(nbytes);
}
inline void my_free (void * ptr, int nbytes)
{
  total_freed += nbytes;
  GC_free (ptr);
}
#else
inline void * my_malloc (int nbytes)
{
  total_allocated += nbytes;
  return malloc(nbytes);
}
inline void my_free (void * ptr, int nbytes)
{
  total_freed += nbytes;
  free (ptr);
}
#endif



/************************************************************/

inline II_Pair * make_ii_pair (IMM i1, IMM i2)
{
  II_Pair *new = (II_Pair *) my_malloc (sizeof(II_Pair));
  new->i1 = i1;
  new->i2 = i2;
  return new;
}


inline IP_Pair * make_ip_pair (IMM i1, void * p2)
{
  IP_Pair *new = (IP_Pair *) my_malloc (sizeof(IP_Pair));
  new->i1 = i1;
  new->p2 = p2;
  return new;
}


inline PP_Pair * make_pp_pair (void * p1, void * p2)
{
  PP_Pair *new = (PP_Pair *) my_malloc (sizeof(PP_Pair));
  new->p1 = p1;
  new->p2 = p2;
  return new;
}


inline IPP_Triple * make_ipp_triple (IMM i1, void * p2, void * p3)
{
  IPP_Triple *new = (IPP_Triple *) my_malloc (sizeof(IPP_Triple));
  new->i1 = i1;
  new->p2 = p2;
  new->p3 = p3;
  return new;
}







/************************************************************/



inline IP_Pair * ilist_cons (IMM val, IP_Pair *inlist)
{ 
  IP_Pair *newpair; 
  newpair = (IP_Pair *) my_malloc (sizeof(IP_Pair)); 
  ICAR(newpair) = val;
  CDR(newpair) = inlist;
  return(newpair); 
} 
 

void ilist_dispose (IP_Pair *inlist)
{ 
  IP_Pair *the_rest; 
  if (inlist == NULL) return; 
  the_rest = CDR(inlist);
  my_free(inlist, sizeof(IP_Pair)); 
  ilist_dispose(the_rest); 
} 

 
int ilist_length (IP_Pair *inlist)
{ 
  IP_Pair *idpt = inlist; 
  int the_length = 0; 
  while (idpt != NULL) { 
    the_length += 1; 
    idpt = CDR(idpt); 
  } 
  return(the_length); 
} 
 

IMM ilist_nth (IP_Pair *inlist, int n)
{ 
  IP_Pair *idpt = inlist; 
  int my_counter = n; 
  while (my_counter > 0) { 
    if (idpt == NULL) exit(-1); 
    idpt = CDR(idpt);
    my_counter -= 1; 
  } 
  if (idpt == NULL) exit(-1); 
  return(ICAR(idpt));
} 
 

IP_Pair * ilist_reverse_aux (IP_Pair *the_list, IP_Pair *new_list)
{ 
  if (the_list == NULL) return new_list; 
  else return
	 (ilist_reverse_aux 
	  (CDR(the_list),
	   ilist_cons (ICAR(the_list), new_list))); 
} 
 
 
IP_Pair * ilist_reverse (IP_Pair *inlist)
{ 
  return(ilist_reverse_aux (inlist, (IP_Pair *) NULL)); 
} 



IP_Pair * ilist_append_bang (IP_Pair * a, IP_Pair * b)
{
  IP_Pair * lastp;
  if (a == NULL) return b;
  else if (b == NULL) return a;
  else {
    lastp = a;
    while (CDR(lastp) != NULL) lastp = CDR(lastp);
    CDR(lastp) = b;
  }
  return a;
}


boolean ilist_member_p (IMM x, IP_Pair * l) 
{
  if (l == NULL) return false;
  else if (x == ICAR (l)) return true;
  else return (ilist_member_p (x, CDR(l)));
}



/***********************/


inline PP_Pair * plist_cons (void * val, PP_Pair *inlist)
{ 
  PP_Pair *newpair; 
  newpair = (PP_Pair *) my_malloc (sizeof(PP_Pair)); 
  PCAR(newpair) = val;
  CDR(newpair) = inlist;
  return(newpair); 
} 
 

void plist_dispose (PP_Pair *inlist)
{ 
  PP_Pair *the_rest; 
  if (inlist == NULL) return; 
  the_rest = CDR(inlist);
  my_free(inlist, sizeof(PP_Pair)); 
  plist_dispose(the_rest); 
} 

 
int plist_length (PP_Pair *inlist)
{ 
  PP_Pair *idpt = inlist; 
  int the_length = 0; 
  while (idpt != NULL) { 
    the_length += 1; 
    idpt = CDR(idpt); 
  } 
  return(the_length); 
} 
 

void * plist_nth (PP_Pair *inlist, int n)
{ 
  PP_Pair *idpt = inlist; 
  int my_counter = n; 
  while (my_counter > 0) { 
    if (idpt == NULL) exit(-1); 
    idpt = CDR(idpt);
    my_counter -= 1; 
  } 
  if (idpt == NULL) exit(-1); 
  return(PCAR(idpt));
} 
 

PP_Pair * plist_reverse_aux (PP_Pair *the_list, PP_Pair *new_list)
{ 
  if (the_list == NULL) return new_list; 
  else return
	 (plist_reverse_aux 
	  (CDR(the_list),
	   plist_cons (PCAR(the_list), new_list))); 
} 
 
 
PP_Pair * plist_reverse (PP_Pair *inlist)
{ 
  return(plist_reverse_aux (inlist, (PP_Pair *) NULL)); 
} 



PP_Pair * plist_append_bang (PP_Pair * a, PP_Pair * b)
{
  PP_Pair * lastp;
  if (a == NULL) return b;
  else if (b == NULL) return a;
  else {
    lastp = a;
    while (CDR(lastp) != NULL) lastp = CDR(lastp);
    CDR(lastp) = b;
  }
  return a;
}


boolean plist_member_p (void * x, PP_Pair * l) 
{
  if (l == NULL) return false;
  else if (x == PCAR (l)) return true;
  else return (plist_member_p (x, CDR(l)));
}



/****************************************************************/


void deallocate_int_list_list (PP_Pair *inlist)
{ 
  IP_Pair *first_list;
  PP_Pair *the_rest; 
  if (inlist == NULL) return; 
  first_list = PCAR(inlist);
  ilist_dispose(first_list);
  the_rest = CDR(inlist);
  my_free(inlist, sizeof(PP_Pair)); 
  deallocate_int_list_list(the_rest); 
} 


void deallocate_list_of_ii (PP_Pair * work_list)
{

  PP_Pair * works;
  do_cdrs (works, work_list) {
    II_Pair * work = PCAR(works);
    my_free (work, sizeof(II_Pair));
  }
  plist_dispose (work_list);
}


void deallocate_list_of_ip (PP_Pair * work_list)
{

  PP_Pair * works;
  do_cdrs (works, work_list) {
    IP_Pair * work = PCAR(works);
    my_free (work, sizeof(IP_Pair));
  }
  plist_dispose (work_list);
}


void deallocate_list_of_pp (PP_Pair * work_list)
{

  PP_Pair * works;
  do_cdrs (works, work_list) {
    PP_Pair * work = PCAR(works);
    my_free (work, sizeof(PP_Pair));
  }
  plist_dispose (work_list);
}


void deallocate_list_of_ipp (PP_Pair * work_list)
{

  PP_Pair * works;
  do_cdrs (works, work_list) {
    IPP_Triple * work = PCAR(works);
    my_free (work, sizeof(IPP_Triple));
  }
  plist_dispose (work_list);
}





/****************************************************************/


I_Stack * make_istack ()
{
  I_Stack *new = (I_Stack *) my_malloc (sizeof(I_Stack));
  new->l = NULL;
  return new;
}

boolean istack_empty_p (I_Stack * s)
{
  return (s->l == NULL);
}

void istack_push (I_Stack * s, IMM obj)
{
  s->l = ilist_cons (obj, s->l);
}

IMM istack_pop (I_Stack * s)
{
  IP_Pair * oldl = s->l;
  IMM val = ICAR(oldl);
  s->l   =   CDR(oldl);   /* deallocate old pair? */
#ifdef USE_FREE
  my_free(oldl,sizeof(IP_Pair));
#endif
  return val;
}


/***********************/


P_Stack * make_pstack ()
{
  P_Stack *new = (P_Stack *) my_malloc (sizeof(P_Stack));
  new->l = NULL;
  return new;
}

boolean pstack_empty_p (P_Stack * s)
{
  return (s->l == NULL);
}

inline void pstack_push (P_Stack * s, void * obj)
{
  s->l = plist_cons (obj, s->l);
}

void * pstack_pop (P_Stack * s)
{
  PP_Pair * oldl = s->l;
  void * val = PCAR(oldl);
  s->l       =  CDR(oldl);   /* deallocate old pair? */
#ifdef USE_FREE
  my_free(oldl,sizeof(PP_Pair));
#endif
  return val;
}


void deallocate_pstack (P_Stack * x)
{
  plist_dispose (x->l);
  my_free(x, sizeof(P_Stack));
}
void deallocate_istack (I_Stack * x)
{
  ilist_dispose (x->l);
  my_free(x, sizeof(I_Stack));
}



/****************************************************************/



I_Queue * make_iqueue ()
{
  I_Queue *new = (I_Queue *) my_malloc (sizeof(I_Queue));
  new->a = NULL;
  new->d = NULL;
  return new;
}

boolean iqueue_empty_p (I_Queue * s)
{
  return (s->a == NULL && s->d == NULL);
}

void iqueue_push (I_Queue * q, IMM obj)
{
  q->a = ilist_cons (obj, q->a);
}


void normalize_iqueue (I_Queue * q)
{
  if (q->d == NULL) {
    q->d = ilist_reverse (q->a);
#ifdef USE_FREE    
    ilist_dispose(q->a);
#endif
    q->a = NULL;
  }
}


/* iqueue_contents calls this */

void aggressively_normalize_iqueue (I_Queue * q)
{
  if (q->a != NULL) {
    q->d = ilist_append_bang (q->d, ilist_reverse (q->a));
#ifdef USE_FREE    
    ilist_dispose(q->a);
#endif
    q->a = NULL;
  }
}




IMM iqueue_pop (I_Queue * q)
{
  IMM head;
  IP_Pair * old_d;
  normalize_iqueue (q);
  head = ICAR(q->d);
  old_d = q->d;
  q->d = CDR(q->d);
#ifdef USE_FREE
  my_free(old_d,sizeof(IP_Pair));
#endif
  return head;
}



void deallocate_iqueue (I_Queue * pq)
{
  ilist_dispose (pq->a);
  ilist_dispose (pq->d);
  my_free(pq, sizeof(I_Queue));
}


IP_Pair * iqueue_contents (I_Queue * s)
{
  aggressively_normalize_iqueue(s);
  return s->d;
}







/****************************/


IP_Pair * istack_contents (I_Stack * s)
{
  return s->l;
}


PP_Pair * pstack_contents (P_Stack * s)
{
  return s->l;
}


/****************************/


P_Queue * make_pqueue ()
{
  P_Queue *new = (P_Queue *) my_malloc (sizeof(P_Queue));
  new->a = NULL;
  new->d = NULL;
  return new;
}

boolean pqueue_empty_p (P_Queue * s)
{
  return (s->a == NULL && s->d == NULL);
}

void pqueue_push (P_Queue * q, void * obj)
{
  q->a = plist_cons (obj, q->a);
}


/* pqueue pop calls this */
void normalize_pqueue (P_Queue * q)
{
  if (q->d == NULL) {
    q->d = plist_reverse (q->a);
#ifdef USE_FREE    
    plist_dispose(q->a);
#endif
    q->a = NULL;
  }
}


/* pqueue_contents calls this */

void aggressively_normalize_pqueue (P_Queue * q)
{
  if (q->a != NULL) {
    q->d = plist_append_bang (q->d, plist_reverse (q->a));
#ifdef USE_FREE    
    plist_dispose(q->a);
#endif
    q->a = NULL;
  }
}


void * pqueue_pop (P_Queue * q)
{
  void * head;
  PP_Pair * old_d;
  normalize_pqueue (q);
  head = PCAR(q->d);
  old_d = q->d;
  q->d = CDR(q->d);
#ifdef USE_FREE
  my_free(old_d,sizeof(PP_Pair));
#endif
  return head;
}



void deallocate_pqueue (P_Queue * pq)
{
  plist_dispose (pq->a);
  plist_dispose (pq->d);
  my_free(pq, sizeof(P_Queue));
}


PP_Pair * pqueue_contents (P_Queue * s)
{
  aggressively_normalize_pqueue(s);
  return s->d;
}






/****************************************************************/





PP_Pair ** make_vector_of_alists (int n)
{
  int i;
  PP_Pair ** new = (PP_Pair **) my_malloc (n * sizeof(PP_Pair *));
  dotimes (i, n) new[i] = NULL;
  return new;
}

#define INITIAL_N_ALISTS 1
#define EXPAND_TABLE_BY  10


IP_Pair * passq (IMM the_key, PP_Pair * inlist)
{
  if (inlist == NULL) return NULL;
  else {
    IP_Pair * the_car = (IP_Pair *) PCAR(inlist);
    if (KEY(the_car) == the_key)
      return (the_car);
    else return (passq (the_key, CDR(inlist)));
  }
}


II_Pair * iassq (IMM the_key, PP_Pair * inlist)
{
  if (inlist == NULL) return NULL;
  else {
    II_Pair * the_car = (II_Pair *) PCAR(inlist);
    if (KEY(the_car) == the_key)
      return (the_car);
    else return (iassq (the_key, CDR(inlist)));
  }
}




/****************************************************************/







void itable_expand (I_Table * table)
{
  int old_n_alists = table->n_alists;
  int new_n_alists = old_n_alists * EXPAND_TABLE_BY;
  PP_Pair ** old_alists = table->alists;
  PP_Pair ** new_alists = make_vector_of_alists (new_n_alists);
  int i;

  dotimes (i, old_n_alists) {
    PP_Pair * al = old_alists [i];
    while (al != NULL) {
      PP_Pair * arest = CDR(al);
      II_Pair * a    = PCAR(al);
      int new_hash = ICAR(a) % new_n_alists;
      new_alists[new_hash] = plist_cons (a, new_alists[new_hash]);
#ifdef USE_FREE
      my_free(al, sizeof(PP_Pair));
#endif
      al = arest;
    }
  }
#ifdef USE_FREE
  my_free(old_alists, old_n_alists * sizeof(PP_Pair *));
#endif
  table->n_alists = new_n_alists;
  table->alists = new_alists;
}





I_Table * make_itable ()
{
  I_Table *new = (I_Table *) my_malloc (sizeof(I_Table));
  new->count = 0;
  new->n_alists = INITIAL_N_ALISTS;
  new->alists = make_vector_of_alists (INITIAL_N_ALISTS);
  return new;
}


boolean itable_test (I_Table * table, int key)
{
  int hash = key % table->n_alists;
  II_Pair * probe = iassq (key, table->alists[hash]);
  return (probe != NULL);
}


IMM itable_lookup (I_Table * table, int key)
{
  int hash = key % table->n_alists;
  II_Pair * probe = iassq (key, table->alists[hash]);
  assert(probe != NULL);
  return(IVAL(probe));
}



void itable_set (I_Table * table, int key, IMM val)
{
  int hash = key % table->n_alists;
  II_Pair * probe = iassq (key, table->alists[hash]);
  if (probe != NULL) IVAL(probe) = val;
  else {
    II_Pair * new = make_ii_pair (key, val);
    table->alists[hash] = plist_cons (new, table->alists[hash]);
    table->count += 1;
    if (table->count > EXPAND_TABLE_BY * table->n_alists)
      itable_expand (table);
  }
}


int itable_count (I_Table * table)
{
  return table->count;
}



void walk_itable (void f(int a, IMM b), I_Table * table)
{
  int i;
  dotimes (i, table->n_alists) {
    PP_Pair * al = table->alists[i];
    while (al != NULL) {
      II_Pair * x = PCAR(al);
      f (KEY(x), IVAL(x));
      al = CDR(al);
    }
  }
}     




void deallocate_itable (I_Table * table)
{
  int i;
  dotimes (i, table->n_alists) {
    PP_Pair * the_list = table->alists[i];
    while (the_list != NULL) {
      PP_Pair * the_rest =  CDR(the_list);
      II_Pair * assoc    = PCAR(the_list);
      my_free(the_list, sizeof(PP_Pair));
      my_free(assoc, sizeof(II_Pair));
      the_list = the_rest;
    }
  }
  my_free(table->alists, table->n_alists * sizeof(PP_Pair *));
  my_free(table, sizeof(I_Table));
}








/****************************************************************/





void ptable_expand (P_Table * table)
{
  int old_n_alists = table->n_alists;
  int new_n_alists = old_n_alists * EXPAND_TABLE_BY;
  PP_Pair ** old_alists = table->alists;
  PP_Pair ** new_alists = make_vector_of_alists (new_n_alists);
  int i;

  dotimes (i, old_n_alists) {
    PP_Pair * al = old_alists [i];
    while (al != NULL) {
      PP_Pair * arest = CDR(al);
      IP_Pair * a    = PCAR(al);
      int new_hash = ICAR(a) % new_n_alists;
      new_alists[new_hash] = plist_cons (a, new_alists[new_hash]);
#ifdef USE_FREE
      my_free(al, sizeof(PP_Pair));
#endif
      al = arest;
    }
  }
#ifdef USE_FREE
  my_free(old_alists, old_n_alists * sizeof(PP_Pair *));
#endif
  table->n_alists = new_n_alists;
  table->alists = new_alists;
}



P_Table * make_ptable ()
{
  P_Table *new = (P_Table *) my_malloc (sizeof(P_Table));
  new->count = 0;
  new->n_alists = INITIAL_N_ALISTS;
  new->alists = make_vector_of_alists (INITIAL_N_ALISTS);
  return new;
}


boolean ptable_test (P_Table * table, int key)
{
  int hash = key % table->n_alists;
  IP_Pair * probe = passq (key, table->alists[hash]);
  return (probe != NULL);
}


void * ptable_lookup (P_Table * table, int key)
{
  int hash = key % table->n_alists;
  IP_Pair * probe = passq (key, table->alists[hash]);
  assert(probe != NULL);
  return(PVAL(probe));
}



void ptable_set (P_Table * table, int key, void * val)
{
  int hash = key % table->n_alists;
  IP_Pair * probe = passq (key, table->alists[hash]);
  if (probe != NULL) PVAL(probe) = val;
  else {
    IP_Pair * new = make_ip_pair (key, val);
    table->alists[hash] = plist_cons (new, table->alists[hash]);
    table->count += 1;
    if (table->count > EXPAND_TABLE_BY * table->n_alists)
      ptable_expand (table);
  }
}


int ptable_count (P_Table * table)
{
  return table->count;
}



void walk_ptable (void f(int a, void * b), P_Table * table)
{
  int i;
  dotimes (i, table->n_alists) {
    PP_Pair * al = table->alists[i];
    while (al != NULL) {
      IP_Pair * x = PCAR(al);
      f (KEY(x), PVAL(x));
      al = CDR(al);
    }
  }
}     


/* Note: this doesn't free the objects that are stored in the table */


void deallocate_ptable (P_Table * table)
{
  int i;
  dotimes (i, table->n_alists) {
    PP_Pair * the_list = table->alists[i];
    while (the_list != NULL) {
      PP_Pair * the_rest =  CDR(the_list);
      IP_Pair * assoc    = PCAR(the_list);
      my_free(the_list, sizeof(PP_Pair));
      my_free(assoc, sizeof(IP_Pair));
      the_list = the_rest;
    }
  }
  my_free(table->alists, table->n_alists * sizeof(PP_Pair *));
  my_free(table, sizeof(P_Table));
}




boolean itable_test_and_set (I_Table * table, int key)
{
  boolean try = itable_test (table, key);
  if (!try) itable_set (table, key, true);
  return try;
}





Boxed_float * make_boxed_float (float f)
{
  Boxed_float *new = (Boxed_float *) my_malloc (sizeof(Boxed_float));
  new->f = f;
  return new;
}



/****************************************************************/



void print_int_list(IP_Pair *inlist)
{
  IP_Pair *idpt = inlist;
  printf("(");
  while (idpt != NULL) {
    printf("%d ", ICAR(idpt));
    idpt = CDR(idpt);
  }
  printf(")\n");
}





void print_int_list_list (PP_Pair *inlist)
{
  PP_Pair *pdpt = inlist;
  printf("(");
  while (pdpt != NULL) {
    IP_Pair *idpt = PCAR(pdpt);
    printf("(");
    while (idpt != NULL) {
      printf("%d ", ICAR(idpt));
      idpt = CDR(idpt);
    }
    printf(")\n");
    pdpt = CDR(pdpt);
  }
  printf(")\n");
}






void print_boxed_float_list(PP_Pair *inlist)
{
  PP_Pair *idpt = inlist;
  printf("(");
  while (idpt != NULL) {
    printf("%f ", ((Boxed_float *) PCAR(idpt))->f);
    idpt = CDR(idpt);
  }
  printf(")\n");
}


/*********************************************************************/





PP_Pair * plist_safe_head_aux (PP_Pair *the_list,
			  PP_Pair *new_list,
			  int counter, int how_many)
{ 
  PP_Pair * result;
  if (counter == how_many || the_list == NULL) {
    result = plist_reverse(new_list); 
#ifdef USE_FREE
    plist_dispose(new_list);
#endif
    return result;
  }
  else return (plist_safe_head_aux 
	       (CDR(the_list),
		plist_cons (PCAR(the_list), new_list),
		counter + 1,
		how_many));
}
 
 
PP_Pair * plist_safe_head (PP_Pair *inlist, int how_many)
{ 
  return(plist_safe_head_aux (inlist, (PP_Pair *) NULL, 0, how_many)); 
} 


IP_Pair * ilist_safe_head_aux (IP_Pair *the_list,
			  IP_Pair *new_list,
			  int counter, int how_many)
{ 
  IP_Pair * result;
  if (counter == how_many || the_list == NULL) {
    result = ilist_reverse(new_list); 
#ifdef USE_FREE
    ilist_dispose(new_list);
#endif
    return result;
  }
  else return (ilist_safe_head_aux 
	       (CDR(the_list),
		ilist_cons (ICAR(the_list), new_list),
		counter + 1,
		how_many));
}
 
 
IP_Pair * ilist_safe_head (IP_Pair *inlist, int how_many)
{ 
  return(ilist_safe_head_aux (inlist, (IP_Pair *) NULL, 0, how_many)); 
} 




/****************************************************************/


#ifdef TEST_APPEND

main()
{
  IP_Pair *a, *b, *c;
  a = ilist_cons(1, ilist_cons(2, ilist_cons (3, NULL)));
  b = ilist_cons(10, ilist_cons(20, ilist_cons (30, NULL)));
  c = ilist_append_bang (a, b);
  print_int_list (c);
}

#endif


#ifdef TEST_IQUEUE_CONTENTS
main ()
{
  I_Queue * a;
  a = make_iqueue ();
  iqueue_push (a, 1);
  iqueue_push (a, 2);
  iqueue_push (a, 3);
  print_int_list (iqueue_contents(a));
  iqueue_push (a, 4);
  iqueue_push (a, 5);
  iqueue_push (a, 6);
  print_int_list (iqueue_contents(a));
}
#endif


#ifdef TEST_ILISTS


char charworkbuf [100];

void get_int_defaut(prompt, varble)
char *prompt;
int *varble;
{
  int response, matched;

  printf("%s [y for %d] ", prompt, *varble);
  scanf ("%s", charworkbuf);
  if (charworkbuf[0] == 'y') return;
  matched = sscanf(charworkbuf, "%d", &response);
  *varble = response;
}


main()
{
  IP_Pair *the_list = NULL;
  IMM newval;

  while (1) {
    printf("%d: ", ilist_length(the_list));
    print_int_list(the_list);
    print_int_list(ilist_reverse(the_list));
    get_int_defaut("next value: ", &newval);
    the_list = ilist_cons(newval, the_list);
  }
}
#endif





#ifdef TEST_STACKS_AND_QUEUES

void test_istacks ()
{
  int i,j,k;
  I_Stack * pig = make_istack ();
  printf("\ntesting istacks\n");
  printf("empty? %d\n", istack_empty_p(pig));
  istack_push(pig,1);
  print_int_list (pig->l);
  istack_push(pig,2);
  print_int_list (pig->l);
  istack_push(pig,3);
  print_int_list (pig->l);
  printf("empty? %d\n", istack_empty_p(pig));
  i = istack_pop(pig);
  j = istack_pop(pig);
  k = istack_pop(pig);
  print_int_list (pig->l);
  printf("asdf %d %d %d\n", i,j,k);
  printf("empty? %d\n", istack_empty_p(pig));
}

void test_pstacks ()
{
  float i,j,k;
  P_Stack * pig = make_pstack ();
  printf("\ntesting pstacks\n");
  printf("empty? %d\n", pstack_empty_p(pig));
  pstack_push(pig,make_boxed_float(1.0));
  print_boxed_float_list (pig->l);
  pstack_push(pig,make_boxed_float(2.0));
  print_boxed_float_list (pig->l);
  pstack_push(pig,make_boxed_float(3.0));
  print_boxed_float_list (pig->l);
  printf("empty? %d\n", pstack_empty_p(pig));
  i = ((Boxed_float *) pstack_pop(pig))->f;
  j = ((Boxed_float *) pstack_pop(pig))->f;
  k = ((Boxed_float *) pstack_pop(pig))->f;
  print_boxed_float_list (pig->l);
  printf("asdf %f %f %f\n", i,j,k);
  printf("empty? %d\n", pstack_empty_p(pig));
}

/*
void test_iqueues ()
{
  int i,j,k;
  I_Queue * pig = make_iqueue ();
  printf("\ntesting iqueues\n");
  printf("empty? %d\n", iqueue_empty_p(pig));
  iqueue_push(pig,1);
  print_int_list (pig->l);
  iqueue_push(pig,2);
  print_int_list (pig->l);
  iqueue_push(pig,3);
  print_int_list (pig->l);
  printf("empty? %d\n", iqueue_empty_p(pig));
  i = iqueue_pop(pig);
  j = iqueue_pop(pig);
  k = iqueue_pop(pig);
  print_int_list (pig->l);
  printf("asdf %d %d %d\n", i,j,k);
  printf("empty? %d\n", iqueue_empty_p(pig));
}
*/


void test_pqueues ()
{
  float i,j,k;
  P_Queue * pig = make_pqueue ();
  printf("\ntesting pqueues\n");
  printf("empty? %d\n", pqueue_empty_p(pig));
  pqueue_push(pig,make_boxed_float(1.0));
  print_boxed_float_list (pig->a);  print_boxed_float_list (pig->d);
  pqueue_push(pig,make_boxed_float(2.0));
  print_boxed_float_list (pig->a);  print_boxed_float_list (pig->d);
  pqueue_push(pig,make_boxed_float(3.0));
  print_boxed_float_list (pig->a);  print_boxed_float_list (pig->d);
  print_boxed_float_list (pqueue_contents(pig));
  print_boxed_float_list (pig->a);  print_boxed_float_list (pig->d);
  printf("empty? %d\n", pqueue_empty_p(pig));
  i = ((Boxed_float *) pqueue_pop(pig))->f;
  j = ((Boxed_float *) pqueue_pop(pig))->f;
  k = ((Boxed_float *) pqueue_pop(pig))->f;

  print_boxed_float_list (pig->a);  print_boxed_float_list (pig->d);

  printf("asdf %f %f %f\n", i,j,k);
  printf("empty? %d\n", pqueue_empty_p(pig));
}


int main()
{
  test_istacks ();
  test_pstacks ();
  /*  test_iqueues (); */
  test_pqueues ();
  return(0);
}




#endif






#ifdef TEST_ITABLES

void print2ints (int a, int b)
{
  printf("%d %d\n", a, b);
}


int main ()
{
  int j;
  int key, val;
  I_Table * pig = make_itable ();

  while (1) {
    walk_itable (print2ints, pig);
    printf("%d ", itable_count(pig));
    dotimes (j, 10) {
      boolean tt = itable_test(pig, j);
      printf (" :%d %d", j, tt);
      if (tt) {
	int v = itable_lookup (pig, j);
	printf("[%d]", v);
      }
    }
    printf("\n");
    printf("new key and value: ");
    scanf("%d %d", &key, &val);
    itable_set(pig, key, val);
  }
  return(0);
}


#endif



#ifdef TEST_PTABLES


void print_int_and_boxed_float (int a, void * bb)
{
  Boxed_float * b = bb;
  printf("%d %f\n", a, b->f);
}










int main ()
{
  int j;
  int key;
  float val;
  P_Table * pig = make_ptable ();

  while (1) {
    walk_ptable (print_int_and_boxed_float, pig);
    printf("%d(%d) ", ptable_count(pig), pig->n_alists);
    dotimes (j, 10) {
      boolean tt = ptable_test(pig,j);
      printf (" :%d %d", j, tt);
      if (tt) {
	Boxed_float * v = ptable_lookup (pig,j);
	printf("[%f]", v->f);
      }
    }
    printf("\n");
    printf("new key and value: ");
    scanf("%d %f", &key, &val);
    ptable_set(pig, key, make_boxed_float(val));
  }
  return(0);
}

#endif



